<?php

namespace App\Http\Controllers\Backend;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\BrandCategory;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use App\Models\Catalogue;

class BrandController extends Controller
{
    public function index(Request $request)
    {
        $query = !empty($request->get('q')) ? $request->get('q') : null;
        $brandsQuery = Brand::with('category');
        if ($query) {
            $brandsQuery = $brandsQuery->search($query, null, true);
        }
        $brands = $brandsQuery->paginate(20);
        return view('backend.brands.index', compact('brands'));
    }

    public function create()
    {
        $categories = BrandCategory::pluck('name', 'id');
        return view('backend.brands.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $brand = Brand::create($request->only(['name', 'description', 'category_id']));
        $brand->slug = str_slug($request->input('name'), '-');
        $brand->save();

        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $path = 'public/brands/';
            $path = $request->file('logo')->storeAs($path, $brand->id . '.' . $file->getClientOriginalExtension());
            $brand->logo = $brand->id . '.' . $file->getClientOriginalExtension();
            $brand->save();
        }

        return redirect()->route('backend.brands.show', $brand->id);
    }

    public function show($id)
    {
        $brand = Brand::findOrFail($id);
        $catalogues =  $brand->catalogues;

        return view('backend.brands.show', compact('brand', 'catalogues'));
    }

    public function edit($id)
    {
        $brand = Brand::findOrFail($id);
        $categories = BrandCategory::pluck('name', 'id');
        return view('backend.brands.edit', compact('brand', 'categories'));
    }

    public function update(Request $request, $id)
    {
        $brand = Brand::findOrFail($id);
        $brand->fill($request->only(['name', 'description', 'category_id']));
        $brand->save();

        if ($request->hasFile('logo')) {
            // Delete old file
            $path = 'public/brands/' . $brand->logo;
            if (!empty($brand->logo) && File::exists(storage_path('app/' . $path))) {
                Storage::delete($path);
            }
            // Upload new file
            $file = $request->file('logo');
            $path = 'public/brands/';
            $path = $request->file('logo')->storeAs($path, $brand->id . '.' . $file->getClientOriginalExtension());
            $brand->logo = $brand->id . '.' . $file->getClientOriginalExtension();
            $brand->save();
        }

        return redirect()->route('backend.brands.show', $brand->id);
    }

    public function delete($id)
    {
        $brand = Brand::find($id);
        $brand->delete();

        return redirect()->route('backend.brands.index');
    }


    public function CatalogueCreate($id)
    {
        $brand = Brand::findOrFail($id);
        return view('backend.brands.catalogues.create', compact('brand'));
    }

    public function CatalogueStore(Request $request, $id)
    {
        $this->validate($request, [
            'file' => 'mimes:pdf',
            'image' => 'mimes:jpeg,png,jpg',
        ], [
            'file.mimes' => 'Le catalogue doit être au format pdf',
            'image.mimes' => 'La photo doit être au format png/jpeg/jpg'
        ]);


        $catalogue = Catalogue::create($request->input());
        $catalogue->slug = str_slug($request->input('name'), '-');
        $catalogue->save();

        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $path = 'public/brands/' . $catalogue->brand_id . '/catalogues/';
            $path = $request->file('file')->storeAs($path, $file->getClientOriginalName());
            $catalogue->filename = $file->getClientOriginalName();
            $catalogue->save();
        }
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $path = 'public/brands/' . $catalogue->brand_id . '/catalogues/';
            $path = $request->file('image')->storeAs($path, $file->getClientOriginalName());
            $catalogue->image = $file->getClientOriginalName();
            $catalogue->save();
        }

        return redirect()->route('backend.brands.show', $catalogue->brand_id);
    }

    public function CatalogueEdit($id, $catalogue_id)
    {
        $brand = Brand::findOrFail($id);
        $catalogue =  Catalogue::findOrFail($catalogue_id);
        return view('backend.brands.catalogues.edit', compact('brand', 'catalogue'));
    }

    public function CatalogueUpdate(Request $request, $id, $catalogue_id)
    {
        $catalogue = Catalogue::findOrFail($catalogue_id);
        $catalogue->fill($request->input());
        $catalogue->save();

        if ($request->hasFile('file')) {
            // Delete old file
            $path = 'public/brands/' . $id . '/catalogues/' . $catalogue->filename;
            if ($catalogue->file && File::exists(storage_path('app/' . $path))) {
                Storage::delete($path);
            }
            // Upload new file
            $file = $request->file('file');
            $path = 'public/brands/' . $id . '/catalogues/';
            $path = $request->file('file')->storeAs($path, $file->getClientOriginalName());
            $catalogue->filename = $file->getClientOriginalName();
            $catalogue->save();
        }
        if ($request->hasFile('image')) {
            // Delete old file
            $path = 'public/brands/' . $id . '/catalogues/' . $catalogue->image;
            if ($catalogue->image && File::exists(storage_path('app/' . $path))) {
                Storage::delete($path);
            }
            // Upload new file
            $file = $request->file('image');
            $path = 'public/brands/' . $id . '/catalogues/';
            $path = $request->file('image')->storeAs($path, $file->getClientOriginalName());
            $catalogue->image = $file->getClientOriginalName();
            $catalogue->save();
        }

        return redirect()->route('backend.brands.show', $catalogue->brand_id);
    }

    public function CatalogueDelete($id, $catalogue_id)
    {
        $catalogue =  Catalogue::findOrFail($catalogue_id);

        $path = 'public/brands/' . $id . '/catalogues/' . $catalogue->filename;
        if (File::exists(storage_path('app/' . $path))) {
            Storage::delete($path);
        }

        $catalogue->delete();

        return redirect()->route('backend.brands.show', $id);
    }

    public function products(Request $request, $id)
    {
        $query = !empty($request->get('q')) ? $request->get('q') : null;
        $brand = Brand::findOrFail($id);
        if ($query) {
            $products = $brand->products()->with(['category', 'brand'])->search($query, null, true)->paginate(10);
        } else {
            $products = $brand->products()->with(['category', 'brand'])->paginate(10);
        }
        return view('backend.brands.products.index', compact('brand', 'products'));
    }
}
